package C4LJ::Article;

# Eric Lease Morgan <emorgan@nd.edu>

# May 28, 2008 - changed issue to an integer-only return value

use C4LJ::DB;
use Carp qw( croak );
use strict;

=head1 NAME 

C4LJ::Article -  A class for representing C4LJ articles

=head1 SYNOPSIS

  use C4LJ::Article;
  foreach ( C4LJ::Article->get_articles ) {
    print '        ID: ' . $_->id       . "\n";
    print '     Title: ' . $_->title    . "\n";
    print '       URL: ' . $_->url      . "\n";
    print '  Abstract: ' . $_->abstract . "\n";
    print '    Author: ' . $_->author   . "\n";
    print '      Date: ' . $_->date     . "\n";
    print '     Issue: ' . $_->issue    . "\n";
    print "\n";
  }

=head1 DESCRIPTION

Use this class to read from the C4LJ (WordPress) database.

=head1 METHODS

=head2 new

This method creates a new, empty article object, or called with an optional id returns a specific article complete with its metadata:

  # create a new empty article object
  my $article = C4LJ::Article->new();
   
  # returns filled article object
  my $article = C4LJ::Article->new( id => 3 );

=head2 id

Returns the database key for this article.

  $id = $article->id

=head2 url

Returns the URL where this article can be read.

  $url = $article->url

=head2 issue

Returns the issue number of this article

  $issue = $article->issue

=head2 title

Returns the title of the article

  $title = $article->title

=head2 abstract

Returns the abstract of the article

  $abstract = $article->abstract

=head2 date

Returns the date when the article was published

  $date = $article->date

=head2 author

Returns the author(s) of the article

  $author = $article->author

=head2 get_articles

This is the heart of the matter. Use this method to get a list (array) of article objects.

  # get all articles
  @articles = C4LJ::Articles->get_articles

Through the use of the issue attribute a list of article objects from that issue will be returned:

  # articles from issue #1
  @issue_01 = C4LJ::Articles->get_articles( issue => 1 );
  
Through the use of the tag attribute a list of article objects that have been tagged with a given string will be returned:

  # articles tagged as book reviews
  @book_reviews = C4LJ::Articles->get_articles( tag => 'book review' );
  
You can not combine the issue and tag attributes in the same call to get_articles. The method is not that smart. 

=head1 AUTHOR

Eric Lease Morgan <emorgan@nd.edu>

=cut

sub new {

	# declare local variables
	my ($class, %opts) = @_;
	my $self = {};

	# check for an id
	if ( $opts{ id }) {
	
		# build a query but return only records classified as issues
		my $query = "SELECT p.*, t.name
                     FROM wp_posts AS p, wp_terms AS t, wp_term_relationships AS r 
                     WHERE ID = ?
                     AND r.term_taxonomy_id = t.term_id
                     AND p.ID = r.object_id
                     AND t.name like 'Issue%'";
		my $dbh = C4LJ::DB->dbh();
		my $rv = $dbh->selectrow_hashref( $query, undef, $opts{ id });
		if ( ref( $rv ) eq "HASH" ) { $self = $rv }
		else { return }
		
	}
	
	# return the object
	return bless $self, $class;
	
}


sub id {

	my $self = shift;
	return $self->{ ID };

}


sub url {

	my $self = shift;
	return $self->{ guid };

}

sub issue {

	my $self = shift;
	my $issue = $self->{ name };
	$issue =~ s/Issue //;
	return $issue;

}


sub title {

	my $self = shift;
	return $self->{ post_title };

}


sub abstract {

	my $self = shift;
	return $self->{ post_excerpt };

}

sub date {

	my $self = shift;
	my $rv = $self->{ post_date };
	$rv =~ s/ .+$//;
	return $rv;

}


sub author {

	my $self = shift;
	my $dbh = C4LJ::DB->dbh();
	my $rv = $dbh->selectrow_hashref( "SELECT meta_value FROM wp_postmeta WHERE post_id = ? AND meta_key = 'author'", undef, $self->id );
	if ( ref( $rv ) eq "HASH" ) { $self = $rv }
	else { return }
	return $self->{ meta_value };

}


sub get_articles {

	my $self  = shift;
	my %opts  = @_;
	my @rv    = ();

	# initialize the default query
	my $query = "SELECT ID FROM wp_posts WHERE guid like '%/articles/%'";
		
	# build an optional query based on tag (term) or issue id
	if ( $opts{ tag } or $opts{ issue }) {
	
		my $term = '';
		if ( $opts{ tag }) { $term = $opts{ tag }}
		if ( $opts{ issue }) { $term = 'Issue ' . $opts{ issue }}

		$query = "SELECT p.ID
		          FROM wp_posts AS p, wp_terms AS t, wp_term_relationships AS r
		          WHERE p.guid like '%/articles/%'
		          AND t.name = '$term'
		          AND r.term_taxonomy_id = t.term_id
		          AND p.ID = r.object_id"
	
	}
		
	# execute a query
	my $dbh = C4LJ::DB->dbh();
	my $article_ids = $dbh->selectcol_arrayref( "$query" );
	foreach my $article_id ( @$article_ids ) { push ( @rv, C4LJ::Article->new( id => $article_id )) }	
	return @rv;
	
}


# return true, or else
1;
