#!/usr/bin/perl

# manage-resources.pl - a sample MyLibrary script to create, find, edit, and delete resources

# Eric Lease Morgan <emorgan@nd.edu>

# 2006-01-11 - wrote pod
# 2003-10-10 - first cut (still embarrassing)

=head1 NAME

manage-resources.pl - create, find, edit, and delete MyLibrary resources as well as batch import, export, index, and search resources

=head1 DESCRIPTION

The top half of this script allows you to do the normal things with any database record: create, find, edit, and delete. In this regard it builds on both the manage-terms.pl and manage-locations.pl scripts because it uses the results of those scripts as controlled vocabulary files for describing resources.

This script provides only limited functionalty when it comes to describing resources; it only allows you to add names and notes to resources when they may have many more qualities. In fact, resources can be described with the full basic Dublin Core elements. See the MyLibrary::Resource module for more detail. Similarly, resources can have more than one location. For example it might be in print form and therfore have a call number. It might also be Internet-based and therefore a URL. This script does not implment this functionality either but it exists in the module.

The bottom half of the script allows you to do batch importing via OAI, create a report against the MyLibrary database in the form of an RDF file, index the list of resources using swish-e, and finally, search the index thus implementing a full-text find feature. Fun!

This is one of the more complicated scripts in this system. Read it after you understand manage-facets.pl and manage-terms.pl.

=head1 SEE ALSO

manage-terms.pl, manage-locations.pl, manage-facets.pl, index-resources.pl, images2mylibrary.pl, doaj2mylibrary.pl, subroutines.pl, search.pl, resources2rdf.pl

=head1 AUTHOR

Eric Lease Morgan

=cut

# include the necessary modules
use lib '../lib/';
use MyLibrary::Facet;
use MyLibrary::Resource;
use MyLibrary::Resource::Location;
use MyLibrary::Term;
use strict;
require 'subroutines.pl';

# display an introduction
&clearScreen;
print "\nUse this program to create and maintain MyLibrary resources.\n\n";

# initialize menu choice
my $choice eq '';

do {

	# display menu
	print "Resources main menu\n";
	print "===================\n";
	print <<EOM;
[c]reate a resource
[f]ind all resources
display [o]ne resource
[e]dit a resource
[d]elete a resource

import DOA[j] content from OAI repository
import i[m]ages from OAI repository
dump as [r]df
[i]ndex resources
[s]earch resources
  
[q]uit
  
EOM
	print "Which one? "; chop ($choice = <STDIN>);
	
	# branch according to chioice; create a resource
	if ($choice eq 'c') {
	
		# get the input
		print "What is the name of this resource? "; chop (my $name = <STDIN>);
		print "Please describe this resource. "; chop (my $note = <STDIN>);
		
		# create the object and begin to fill it
		my $resource = MyLibrary::Resource->new;
		$resource->name($name);
		$resource->note($note);

		# get all terms
		&listAllTerms;
		print "Enter term ids separated by spaces used to classify this resource. "; chop (my $ids = <STDIN>);
		my @ids = split (/ /, $ids);
		$resource->related_terms(new => [@ids]);
		
		# get all locations
		&listAllLocations;
		print "What kind of location does this resource have? "; chop (my $location_type = <STDIN>);
		print "What is the location of this resource? "; chop (my $location = <STDIN>);
		$resource->add_location(location => $location, location_type => $location_type, location_note => '');

		# save
		$resource->commit;
		
		# echo the result
		&listOneResource ($resource->id);
	
	}
	
	# find all resources
	elsif ($choice eq 'f' ) { &listAllResources	}
	
	# display one resource
	elsif ($choice eq 'o' ) {
	
		print "What is the ID of the resource you want to display? "; chop (my $id = <STDIN>);
		&listOneResource ($id);
		
	}
	
	# edit a resource
	elsif ($choice eq 'e' ) {
	
		# get the resource to edit
		print "What is the ID of the resource you want to edit? "; chop (my $id = <STDIN>);
		
		# get it
		my $resource = MyLibrary::Resource->new (id => $id);
		
		# check for success
		if ($resource ne undef) {
		
			# get new input
			print "Change the name of the resource from " . $resource->name . " to: "; chop (my $name = <STDIN>);
			print "Change the note of the resource from " . $resource->note . " to: "; chop (my $note = <STDIN>);
			$resource->name($name);
			$resource->note($note);
			
			# get all terms
			&listAllTerms;
			print "Enter term ids separated by spaces used to classify this resource. "; chop (my $ids = <STDIN>);
			my @ids = split (/ /, $ids);
			$resource->related_terms(new => [@ids]);

			# get all locations
			&listAllLocations;
			print "What kind of location does this resource have? "; chop (my $location_type = <STDIN>);
			print "What is the location of this resource? "; chop (my $location = <STDIN>);
			$resource->add_location(location => $location, location_type => $location_type, location_note => '');

			# save
			$resource->commit;

			# echo the result
			&listOneResource ($resource->id);
			
		}
		
		# error
		else {
		
			print "The id, $id, does not point to a valid resource. Try find.\n";
			
		}
	
	}
	
	# delete a resource
	elsif ($choice eq 'd' ) {
	
		# get the resource to delete
		print "What is the ID of the resource you want to delete? "; chop (my $id = <STDIN>);
		
		# get it
		my $resource = MyLibrary::Resource->new (id => $id);
		
		# check for success
		if ($resource ne undef) {
		
			# display 
			&listOneResource ($id);
			
			# confirm
			print "Do you really want to delete this record [y/n]? "; chop (my $reply = <STDIN>);
			
			# check reply
			if ($reply eq 'y') {
			
				# delete it
				$resource->delete;
				print "Resource deleted.\n";
				
			}
			
		}
		
		# error
		else { print "The id, $id, does not point to a valid resource. Try find.\n" }

	
	}
	
	# index resources
	elsif ($choice eq 'i' ) { system './index-resources.pl' }

	# import doaj content
	elsif ($choice eq 'j' ) { system './doaj2mylibrary.pl' }

	# import image content
	elsif ($choice eq 'm' ) { system './images2mylibrary.pl' }

	# search resources
	elsif ($choice eq 's' ) { system './search.pl' }

	# dump as rdf
	elsif ($choice eq 'r' ) { system './resources2rdf.pl' }

	# quit
	elsif ($choice eq 'q' ) { print "\nBye, bye\n" }
	
	else { print "Unknown choice: $choice. Returning to menu.\n" }
	
} until $choice eq 'q';


