package Musings::Subject;

use Musings::DB;
use Carp qw(croak);
use strict;

=head1 NAME

Musings::Subject

=head1 SYNOPSIS

  use Musings::Subject;
  my $subject = Musings::Subject->new;
  $subject->subject_term('Origami');
  $subject->subject_note('the art of folding paper');
  $subject->commit;
  my $id = $subject->subject_id;
  $subject = Musings::Subject->new(id => $id);
  $subject = Musings::Subject->new(name => 'Origami');
  my $term  = $subject->subject_term;
  my $note  = $subject->subject_note;
  $subject->delete;
  my @subjects    = Musings::Subject->get_subjects;
  my @subjects    = Musings::Subject->get_subjects(name => 'Origami');
  my @article_ids = Musings::Subject->article_ids;


=head1 DESCRIPTION

This package does the necessary I/O against the underlying database's subjects table.


=head1 METHODS


=head2 new

Creates a new subject object. If passed an id parameter plus a valid subject ID, then a specific subject object will be created. If passed a name parameter, then the returned object will contain data matching the value of the term.


=head2 subject_term

Get and set the the subject term.


=head2 subject_note

Get and set the definition of the subject term.


=head2 commit

Save the current subject object's attributes to the database. Returns true upon success.


=head2 subject_id

Gets the current ID of the current subject object. You can not set the subject_id value.


=head2 delete

Deletes (drops) the current subject from the database. Be careful. Your application is responsible for passing the ACID test.


=head2 get_subjects

Returns an array of all the Musings::Subject objects in the database. The returned objects will be sorted by subject_term. If a name parameter is passed, then only objects like that name will returned in the array.


=head2 author_ids

Return an array of Musings::Article ID's that are linked (joined/related) to the current subject term. In other words, this method returns the keys to the articles that have been classified with the current term.


=head1 AUTHORS

Eric Lease Morgan <eric_morgan@infomotions.com>

=cut


sub new {

	# declare local variables
	my ($class, %opts) = @_;
	my $self = {};

	# check for an id
	if ($opts{id}) {
	
		my $dbh = Musings::DB->dbh();
		my $rv = $dbh->selectrow_hashref('SELECT * FROM subjects WHERE subject_id = ?', undef, $opts{id});
		if (ref($rv) eq "HASH") {
		
			$self = $rv;

			# build a list of article ids
			my @article_ids = ();
			my $ids = $dbh->selectall_arrayref('SELECT article_id FROM items4subjects WHERE subject_id = ?', undef, $opts{id});
			foreach (@{$ids}) { push (@article_ids, $_->[0]) }
			$self->{article_ids} = ([@article_ids]);
		
		}

		else { return }
		
	}
	
	elsif ($opts{name}) {
	
		my $dbh = Musings::DB->dbh();
		my $rv = $dbh->selectrow_hashref('SELECT * FROM subjects WHERE subject_term = ?', undef, $opts{name});
		if (ref($rv) eq "HASH") { $self = $rv }
		else { return } 
		
	}
	
	# return the object
	return bless $self, $class;
	
}


sub subject_id {

	my $self = shift;
	return $self->{subject_id};

}


sub subject_term {

	# declare local variables
	my ($self, $subject_term) = @_;
	
	# check for the existance of a note 
	if ($subject_term) { $self->{subject_term} = $subject_term }
	
	# return the name
	return $self->{subject_term};
	
}

sub subject_note {

	# declare local variables
	my ($self, $subject_note) = @_;
	
	# check for the existance of a note 
	if ($subject_note) { $self->{subject_note} = $subject_note }
	
	# return the name
	return $self->{subject_note};
	
}



sub commit {

	# get myself, :-)
	my $self = shift;
	
	# get a database handle
	my $dbh = Musings::DB->dbh();	
	
	# see if the object has an id
	if ($self->subject_id()) {
	
		# update the record with this id
		my $return = $dbh->do('UPDATE subjects SET subject_term = ?, subject_note = ? WHERE subject_id = ?', undef, $self->subject_term(), $self->subject_note(), $self->subject_id());
		if ($return > 1 || ! $return) { croak "Subject update in commit() failed. $return records were updated." }
	
	}
	
	else {
	
		# get a new sequence
		my $id = Musings::DB->nextID();
		
		# create a new record
		my $return = $dbh->do('INSERT INTO subjects (subject_id, subject_term, subject_note) VALUES (?, ?, ?)', undef, $id, $self->subject_term, $self->subject_note);
		if ($return > 1 || ! $return) { croak 'Subject commit() failed.'; }
		$self->{subject_id} = $id;
		
	}
	
	# done
	return 1;
	
}


sub delete {

	my $self = shift;

	if ($self->{subject_id}) {

		my $dbh = Musings::DB->dbh();

		# now, delete the primary facet record
		my $rv = $dbh->do('DELETE FROM subjects WHERE subject_id = ?', undef, $self->{subject_id});
		if ($rv != 1) {croak ("Error deleting subject record. Deleted $rv records.");}
		return 1;

	}

	return 0;

}


sub article_ids {

	# get the input
	my $self       = shift;
	my @article_ids = @_;
								
	# check for input
	if (@article_ids) { $self->{article_ids} = ([@article_ids]) }
		
	else {
	
		# rebuild author_ids
		@article_ids = ();
		foreach (@{$self->{article_ids}}) { push(@article_ids, $_) }
		
	}
		
	# done
	return @article_ids;
	
}


sub get_subjects {

	my $self = shift;
	my %opts = @_;
	my @rv   = ();

	my ($sort, $field, $value, $sort_clause, $limit_clause, $query);
	if (defined($opts{sort})) {
	
		if ($opts{sort} eq 'name') { $sort_clause = 'ORDER BY subject_term' }
	}
	
	if (defined($opts{field}) && defined($opts{value})) {
		$field = $opts{'field'};
		$value = $opts{'value'};
		if ($field eq 'name') { $limit_clause = "WHERE subject_term LIKE \'%$value%\'" }
		elsif ($field eq 'description') { $limit_clause = "WHERE subject_note LIKE \'%$value%\'" }
	}
	
	$query = 'SELECT subject_id FROM subjects';
	if ($limit_clause) { $query .= " $limit_clause" }
	if ($sort_clause) { $query .= " $sort_clause" }

	# create and execute a query
	my $dbh = Musings::DB->dbh();

	my $facet_ids = $dbh->selectcol_arrayref("$query");
			
	foreach my $facet_id (@$facet_ids) { push (@rv, Musings::Subject->new(id => $facet_id)) }	
	
	return @rv;
}


# return true
1;
