package Musings::Template;

use Musings::DB;
use Carp qw(croak);
use strict;


=head1 NAME

Musings::Template

=head1 SYNOPSIS

  use Musings::Template;
  my $template = Musings::Template->new;
  $template->template_name('Musings on Information...');
  $template->template_content('<!DOCTYPE TEI.2 PUBLIC "-//TEI P4...');
  $template->commit;
  my $id = $template->template_id;
  $template = Musings::Template->new(id => $id);
  my $template_name    = $template->template_name;
  my $template_content = $template->template_content;
  $template->delete;
  my @templates = Musings::Template->get_templates;


=head1 DESCRIPTION

This package does the necessary I/O against the underlying database's templates table.


=head1 METHODS


=head2 new

Creates a new template object. If passed an id parameter plus a valid template ID, then a specific template object will be created.


=head2 template_name

Get and set the template's name.


=head2 template_content

Get and set the content of the template.

This is expected to be skeleton of an XML file of your design. Consider adding tokens to the template like ##AUTHOR## or ##TITLE## so your application can: 1) read your template into memory, 2) get the values for things like author and title, 3) do the necessary substitutions, and 4) save the resulting XML to the file system or HTTP connection.


=head2 commit

Save the current template object's attributes to the database. Returns true upon success.


=head2 template_id

Gets the current ID of the current template object. You can not set the template_id value.


=head2 delete

Deletes (drops) the current template from the database. Be careful. Your application is responsible for passing the ACID test.


=head2 get_templates

Returns an array of all the template objects in the database. The returned objects will be sorted by template_name.


=head1 AUTHORS

Eric Lease Morgan <eric_morgan@infomotions.com>

=cut

sub new {

	# declare local variables
	my ($class, %opts) = @_;
	my $self = {};

	# check for an id
	if ($opts{id}) {
	
		my $dbh = Musings::DB->dbh();
		my $rv = $dbh->selectrow_hashref('SELECT * FROM templates WHERE template_id = ?', undef, $opts{id});
		if (ref($rv) eq "HASH") { $self = $rv }
		else { return }
		
	}
	
	# return the object
	return bless $self, $class;
	
}


sub template_id {

	my $self = shift;
	return $self->{template_id};

}


sub template_name {

	# declare local variables
	my ($self, $template_name) = @_;
	
	# check for the existance of a note 
	if ($template_name) { $self->{template_name} = $template_name }
	
	# return the name
	return $self->{template_name};
	
}


sub template_content {

	# declare local variables
	my ($self, $template_content) = @_;
	
	# check for the existance of a note 
	if ($template_content) { $self->{template_content} = $template_content }
	
	# return the name
	return $self->{template_content};
	
}



sub commit {

	# get myself, :-)
	my $self = shift;
	
	# get a database handle
	my $dbh = Musings::DB->dbh();	
	
	# see if the object has an id
	if ($self->template_id()) {
	
		# update the record with this id
		my $return = $dbh->do('UPDATE templates SET template_name = ?, template_content = ? WHERE template_id = ?', undef, $self->template_name(), $self->template_content(), $self->template_id());
		if ($return > 1 || ! $return) { croak "Template update in commit() failed. $return records were updated." }
	
	}
	
	else {
	
		# get a new sequence
		my $id = Musings::DB->nextID();
		
		# create a new record
		my $return = $dbh->do('INSERT INTO templates (template_id, template_name, template_content) VALUES (?, ?, ?)', undef, $id, $self->template_name, $self->template_content);
		if ($return > 1 || ! $return) { croak 'Template commit() failed.'; }
		$self->{template_id} = $id;
		
	}
	
	# done
	return 1;
	
}


sub delete {

	my $self = shift;

	if ($self->{template_id}) {

		my $dbh = Musings::DB->dbh();

		# now, delete the primary facet record
		my $rv = $dbh->do('DELETE FROM templates WHERE template_id = ?', undef, $self->{template_id});
		if ($rv != 1) {croak ("Error deleting template record. Deleted $rv records.");}
		return 1;

	}

	return 0;

}


sub get_templates {

	my $self = shift;
	my @rv   = ();

	# create and execute a query
	my $dbh = Musings::DB->dbh();
	my $template_ids = $dbh->selectcol_arrayref('SELECT * FROM templates ORDER BY template_name');
	foreach my $template_id (@$template_ids) { push (@rv, Musings::Template->new(id => $template_id)) }	
	
	# done
	return @rv;
	
}


# return true
1;
